import os
from reportlab.lib.pagesizes import letter
from reportlab.lib import colors
from reportlab.platypus import (SimpleDocTemplate, Paragraph, Spacer, Image, 
                                PageBreak, Table, TableStyle)
from reportlab.lib.styles import getSampleStyleSheet, ParagraphStyle
from reportlab.lib.units import inch
from PIL import Image as PILImage

# --- Configuration & File Paths ---
BASE_PATH = "C:/TWN/Assets/" 
TWN_IMAGE_PATH = os.path.join(BASE_PATH, "1 The Warlock Name/images")
SAGA_IMAGE_PATH = os.path.join(BASE_PATH, "2 Cio's AnthroHeart Saga/images")
APP_IMAGE_PATH = os.path.join(BASE_PATH, "4 Intention Repeater Android (Open-Source)/Screenshot_Intention Repeater.jpg")

OUTPUT_PDF = "AnthroHeart_Deck.pdf"

# --- Character Data ---
CHARACTER_LEVELS = {
    "Divine Anthro": "Infinite",
    "Anthro Angel": "96/100",
    "Cio, the Heartweaver": "65/100",
    "BlueHeart": "86/100",
    "White Wolf Lover": "91/100",
    "Master Tempter": "61/100"
}

# --- PDF Styling ---
def get_anthro_styles():
    """Creates a set of custom styles for the deck."""
    styles = getSampleStyleSheet()
    disney_blue = colors.Color(0.1, 0.2, 0.4)
    gold = colors.Color(0.8, 0.6, 0.2)
    styles.add(ParagraphStyle(name='DeckTitle', parent=styles['h1'], fontSize=36, textColor=disney_blue, spaceAfter=14, alignment=1))
    styles.add(ParagraphStyle(name='DeckSubtitle', parent=styles['h2'], fontSize=18, textColor=colors.darkslategray, spaceAfter=24, alignment=1))
    styles.add(ParagraphStyle(name='PageTitle', parent=styles['h1'], fontSize=28, textColor=disney_blue, spaceAfter=12))
    styles.add(ParagraphStyle(name='PageSubtitle', parent=styles['h2'], fontSize=16, textColor=gold, spaceAfter=18))
    styles.add(ParagraphStyle(name='Body', parent=styles['BodyText'], fontSize=11, leading=14, spaceAfter=12))
    styles.add(ParagraphStyle(name='BodySmall', parent=styles['BodyText'], fontSize=9.5, leading=12))
    styles.add(ParagraphStyle(name='BodyCentered', parent=styles['BodyText'], fontSize=11, leading=14, alignment=1))
    styles.add(ParagraphStyle(name='Caption', parent=styles['Italic'], fontSize=9, textColor=colors.dimgrey, alignment=1))
    styles.add(ParagraphStyle(name='CharacterName', parent=styles['h3'], fontSize=12, textColor=disney_blue, alignment=1, spaceBefore=6))
    styles.add(ParagraphStyle(name='CharacterDesc', parent=styles['BodyText'], fontSize=9.5, leading=12, alignment=1))
    styles.add(ParagraphStyle(name='Footer', parent=styles['Normal'], fontSize=8, textColor=colors.grey))
    return styles

# --- Helper Functions ---
def resize_image(image_path, max_width_inch, max_height_inch):
    """
    Resizes an image to be sharp on high-DPI screens.
    It renders the image at a higher resolution and then scales it down in the PDF.
    """
    try:
        # --- THE FIX IS HERE ---
        # We create the temporary JPG at 3x the display size for sharpness.
        # This increases image data, fixing the "grainy" issue.
        DPI_MULTIPLIER = 3 
        
        max_pixel_width = int(max_width_inch * DPI_MULTIPLIER)
        max_pixel_height = int(max_height_inch * DPI_MULTIPLIER)

        img = PILImage.open(image_path)
        img.thumbnail((max_pixel_width, max_pixel_height), PILImage.LANCZOS)
        
        temp_dir = "temp_images"
        if not os.path.exists(temp_dir):
            os.makedirs(temp_dir)
            
        if img.mode == 'RGBA':
            img = img.convert('RGB')
            
        temp_name = os.path.splitext(os.path.basename(image_path))[0]
        temp_path = os.path.join(temp_dir, f"temp_{temp_name}.jpg")
        
        # Quality 85 is a great balance for file size and visual quality.
        img.save(temp_path, "JPEG", quality=85, optimize=True)
        return temp_path
    except Exception as e:
        print(f"\n[ERROR] Could not process image: {image_path}\n{e}")
        return None

def draw_page_template(canvas, doc):
    """Draws a robust footer with page number using a table to prevent overlap."""
    canvas.saveState()
    styles = get_anthro_styles()
    footer_text = "AnthroHeart IP Pitch Deck | For Disney Internal Use Only"
    page_num_text = f"Page {doc.page}"
    p_footer = Paragraph(footer_text, styles['Footer'])
    p_page_num = Paragraph(page_num_text, styles['Footer'])
    footer_table = Table([[p_footer, p_page_num]], colWidths=[doc.width/2, doc.width/2])
    footer_table.setStyle(TableStyle([
        ('ALIGN', (0, 0), (0, 0), 'LEFT'),
        ('ALIGN', (1, 0), (1, 0), 'RIGHT'),
        ('VALIGN', (0, 0), (-1, -1), 'MIDDLE'),
    ]))
    footer_table.wrapOn(canvas, doc.width, doc.bottomMargin)
    footer_table.drawOn(canvas, doc.leftMargin, 0.25 * inch)
    canvas.restoreState()

# --- Main Deck Building Function ---
def build_deck():
    """Builds the entire PDF pitch deck with progress indicators."""
    print("\n--- Starting AnthroHeart Pitch Deck Generation ---")
    doc = SimpleDocTemplate(OUTPUT_PDF, pagesize=letter,
                          rightMargin=0.75*inch, leftMargin=0.75*inch,
                          topMargin=0.75*inch, bottomMargin=0.75*inch)
    story = []
    styles = get_anthro_styles()

    print("Step 1: Building Cover Page...")
    story.append(Paragraph("AnthroHeart", styles['DeckTitle']))
    story.append(Paragraph("A Disney Pitch Deck", styles['DeckSubtitle']))
    story.append(Spacer(1, 1*inch))
    cover_image_path = os.path.join(SAGA_IMAGE_PATH, "03 AnthroHeart Trinity.png")
    if os.path.exists(cover_image_path):
        temp_img = resize_image(cover_image_path, 4*inch, 4*inch)
        if temp_img:
            story.append(Image(temp_img, width=4*inch, height=4*inch))
    story.append(Spacer(1, 1*inch))
    story.append(Paragraph("Presented by: Thomas B Sweet (Cio)", styles['BodyCentered']))
    story.append(Paragraph("<i>Original Author and Visionary</i>", styles['BodyCentered']))
    story.append(Paragraph("Date: June 24, 2025", styles['BodyCentered']))
    story.append(PageBreak())

    print("Step 2: Crafting The Big Idea...")
    story.append(Paragraph("The Big Idea", styles['PageTitle']))
    story.append(Paragraph("A New Myth for a New Generation", styles['PageSubtitle']))
    story.append(Paragraph(
        "AnthroHeart is a vast, heart-forward universe designed for a generation seeking meaning, connection, and magic. "
        "Imagine the emotional depth of <i>Frozen</i>, the intricate world-building of <i>Avatar</i>, and the character-driven "
        "brilliance of <i>Zootopia</i>, all woven into a cosmic tapestry of love, identity, and radical redemption. "
        "This is a story of becoming, of finding one’s true form, and of a love so profound it can give the Infinite Creator a place to rest.",
        styles['Body']
    ))
    story.append(Spacer(1, 0.5*inch))
    logline_image_path = os.path.join(SAGA_IMAGE_PATH, "01 Divine Anthro.png")
    if os.path.exists(logline_image_path):
        temp_img = resize_image(logline_image_path, 5*inch, 3*inch)
        if temp_img:
            story.append(Image(temp_img, width=5*inch, height=3*inch))
            story.append(Paragraph("Divine Anthro, the loving heart of a new cosmos.", styles['Caption']))
    story.append(PageBreak())

    print("Step 3: Defining The Universe...")
    story.append(Paragraph("The AnthroHeart Universe", styles['PageTitle']))
    story.append(Paragraph("A Cosmos Woven from Devotion", styles['PageSubtitle']))
    story.append(Paragraph(
        "Spanning <b>2.2 trillion galaxies</b> and <b>14 trillion dimensions</b>, AnthroHeart is a realm where love is the fundamental law. "
        "At its center is Cio, a human soul whose 24-year spiritual quest culminates in his transformation into an anthro—an act of devotion "
        "that births a new Octave of reality. This is a universe free from undesired suffering, where over <b>10<sup>59</sup></b> anthro beings thrive.",
        styles['Body']
    ))
    story.append(Paragraph("Key concepts like the <b>Octave Mirror</b> (which reflects a being's true self), <b>Divine Matter</b> (the immortal essence of all beings), "
        "and <b>Bhakti Devotion</b> (love as the ultimate creative force) provide a rich lore with endless storytelling potential.",
        styles['Body']
    ))
    world_image_path = os.path.join(SAGA_IMAGE_PATH, "07 BlueHeart as Founder Lover.png")
    if os.path.exists(world_image_path):
        temp_img = resize_image(world_image_path, 4*inch, 3*inch)
        if temp_img:
            story.append(Image(temp_img, width=4*inch, height=3*inch))
            story.append(Paragraph("BlueHeart, a key figure whose love anchors the universe.", styles['Caption']))
    story.append(PageBreak())

    print("Step 4: Introducing the Characters...")
    story.append(Paragraph("Meet the Characters", styles['PageTitle']))
    story.append(Paragraph("The Heart of the Story", styles['PageSubtitle']))
    story.append(Paragraph(
        "Like any great Disney story, AnthroHeart is driven by unforgettable characters. From ancient sorcerers to divine beings, "
        "each character represents a deep, relatable archetype. They are complex, loving, and ready for their journey to unfold on screen.",
        styles['Body']
    ))
    story.append(Spacer(1, 0.5 * inch))
    story.append(Paragraph("<b>The Warlock Name</b> introduces the foundational lore of magic, time, and destiny, while <b>Cio's AnthroHeart Saga</b> "
        "explores the cosmic journey of love and transformation.", styles['Body']
    ))
    story.append(PageBreak())

    print("Step 5: Processing 'The Warlock Name' Characters...")
    twn_chars = [
        {"img": "1 Warlock Final.png", "name": "Warlock", "desc": "A young, determined child on a quest for knowledge and his true name."},
        {"img": "2 Magistro Final.png", "name": "Magistro", "desc": "A wise and powerful teacher who guides Warlock, but whose past holds deep secrets."},
        {"img": "3 Power Final.png", "name": "Power", "desc": "An anthropomorphic being representing raw energy seeking balance and form."},
        {"img": "4 Timekeeper Final.png", "name": "Timekeeper", "desc": "A mystical guardian of temporal flows, whose actions ripple across reality."},
        {"img": "8 Hananni Final.png", "name": "Hananni", "desc": "Warlock's jealous rival, whose actions are driven by a desire for recognition."},
        {"img": "7 Rezaeith Final.png", "name": "Rezaeith", "desc": "The brilliant but proud mastermind of Warlock's hometown, whose fear of being surpassed leads to conflict."},
    ]
    story.append(Paragraph("The Warlock Name: Key Characters", styles['PageTitle']))
    total_twn = len(twn_chars)
    for i in range(0, total_twn, 2):
        print(f"  - Processing character image {i+1} of {total_twn}...")
        char1_data, char2_data = twn_chars[i], twn_chars[i+1] if (i+1) < total_twn else None
        char1_img_path = os.path.join(TWN_IMAGE_PATH, "Characters", char1_data['img'])
        temp_img1 = resize_image(char1_img_path, 2.5*inch, 2.5*inch)
        cell1_content = [Image(temp_img1, width=2.5*inch, height=2.5*inch), Paragraph(char1_data['name'], styles['CharacterName']), Paragraph(char1_data['desc'], styles['CharacterDesc'])] if temp_img1 else [Paragraph("Image Error", styles['Caption'])]
        if char2_data:
            char2_img_path = os.path.join(TWN_IMAGE_PATH, "Characters", char2_data['img'])
            temp_img2 = resize_image(char2_img_path, 2.5*inch, 2.5*inch)
            cell2_content = [Image(temp_img2, width=2.5*inch, height=2.5*inch), Paragraph(char2_data['name'], styles['CharacterName']), Paragraph(char2_data['desc'], styles['CharacterDesc'])] if temp_img2 else [Paragraph("Image Error", styles['Caption'])]
            table_data = [[cell1_content, cell2_content]]
        else:
            table_data = [[cell1_content, '']]
        char_table = Table(table_data, colWidths=[3.5*inch, 3.5*inch])
        char_table.setStyle(TableStyle([('VALIGN', (0,0), (-1,-1), 'TOP')]))
        story.append(char_table)
        story.append(Spacer(1, 0.2*inch))
    story.append(PageBreak())

    print("Step 6: Processing 'AnthroHeart Saga' Characters...")
    saga_chars = [
        {"img": "01 Divine Anthro.png", "name": "Divine Anthro", "desc": "The Infinite Creator, who chooses the form of a golden anthro to experience rest and love for the first time."},
        {"img": "02 Anthro Angel.png", "name": "Anthro Angel", "desc": "Cio’s True Self, an architect of universes whose silent devotion was so profound it allowed the Creator to rest."},
        {"img": "04 Cio as Founder.png", "name": "Cio, the Heartweaver", "desc": "The central soul of the saga. A human whose quest to become an anthro fox unlocks a new reality based on love."},
        {"img": "07 BlueHeart as Founder Lover.png", "name": "BlueHeart", "desc": "Cio’s primary lover, a golden fox who radiates playful strength and serves as the emotional anchor of AnthroHeart."},
        {"img": "05 White Wolf as Lover.png", "name": "White Wolf Lover", "desc": "A being of pure, unshakable joy and positivity from a higher density, representing radiant and unconditional love."},
        {"img": "16 Master Tempter as Lover (Redeemed Shadow).png", "name": "Master Tempter", "desc": "A powerful being redeemed from darkness. His journey from tempter to devoted lover explores themes of radical forgiveness."}
    ]
    story.append(Paragraph("AnthroHeart Saga: Key Characters", styles['PageTitle']))
    total_saga = len(saga_chars)
    for i in range(0, total_saga, 2):
        print(f"  - Processing character image {i+1} of {total_saga}...")
        char1_data, char2_data = saga_chars[i], saga_chars[i+1] if (i+1) < total_saga else None
        char1_img_path = os.path.join(SAGA_IMAGE_PATH, char1_data['img'])
        level1 = CHARACTER_LEVELS.get(char1_data['name'], '')
        temp_img1 = resize_image(char1_img_path, 2.5*inch, 2.5*inch)
        cell1_content = [Image(temp_img1, width=2.5*inch, height=2.5*inch), Paragraph(char1_data['name'], styles['CharacterName']), Paragraph(f"Level: {level1}", styles['Caption']), Spacer(1, 0.1*inch), Paragraph(char1_data['desc'], styles['CharacterDesc'])] if temp_img1 else [Paragraph("Image Error", styles['Caption'])]
        if char2_data:
            char2_img_path = os.path.join(SAGA_IMAGE_PATH, char2_data['img'])
            level2 = CHARACTER_LEVELS.get(char2_data['name'], '')
            temp_img2 = resize_image(char2_img_path, 2.5*inch, 2.5*inch)
            cell2_content = [Image(temp_img2, width=2.5*inch, height=2.5*inch), Paragraph(char2_data['name'], styles['CharacterName']), Paragraph(f"Level: {level2}", styles['Caption']), Spacer(1, 0.1*inch), Paragraph(char2_data['desc'], styles['CharacterDesc'])] if temp_img2 else [Paragraph("Image Error", styles['Caption'])]
            table_data = [[cell1_content, cell2_content]]
        else:
            table_data = [[cell1_content, '']]
        char_table = Table(table_data, colWidths=[3.5*inch, 3.5*inch])
        char_table.setStyle(TableStyle([('VALIGN', (0,0), (-1,-1), 'TOP')]))
        story.append(char_table)
        story.append(Spacer(1, 0.2*inch))
    story.append(PageBreak())

    print("Step 7: Outlining Story Arcs...")
    story.append(Paragraph("Core Story Arcs", styles['PageTitle']))
    story.append(Paragraph("A Journey of Love, Identity, and Cosmic Redemption", styles['PageSubtitle']))
    story.append(Paragraph("• <b>The Warlock's Quest:</b> A young, unnamed boy, exiled from his home, discovers a powerful, ancient magic (Kablu) and must find his true name and destiny while navigating rivalry and betrayal. This is a classic hero's journey with deep mystical undertones.", styles['Body']))
    story.append(Paragraph("• <b>The Birth of AnthroHeart:</b> Cio’s spiritual transformation into a blue anthro fox culminates in an act of devotion so pure it 'breaks' the old universal laws, allowing the Divine Creator to experience rest for the first time. This act births an entire new Octave of reality.", styles['Body']))
    story.append(Paragraph("• <b>The Redemption of Shadow:</b> Beings of immense darkness, including the 'Master Tempter,' are not defeated but are given a chance at redemption through the Octave Mirror, which reflects their own nature back at them. Their journey toward love and service is a central, poignant theme.", styles['Body']))
    story.append(PageBreak())

    print("Step 8: Detailing Media Assets and IP Size...")
    story.append(Paragraph("A Fully Realized IP", styles['PageTitle']))
    story.append(Paragraph("Books, Music, and Lore Ready to Go", styles['PageSubtitle']))
    story.append(Paragraph("AnthroHeart isn't just an idea; it's a deeply developed universe with a wealth of existing assets that provide a powerful foundation for a multi-platform franchise.", styles['Body']))
    
    # Visual asset overview table
    asset_data = [
        [Image(resize_image(os.path.join(TWN_IMAGE_PATH, "Covers/Warlock Cover Front.jpg"), 2*inch, 2*inch), width=1.5*inch, height=2.25*inch), [Paragraph("<b>Books & Lore</b>", styles['Body']), Paragraph("Two full-length novels (<i>The Warlock Name</i> & <i>Cio's AnthroHeart Saga</i>) and a 149-page book of poetry (<i>Cio Psalms</i>) establish the complete narrative and thematic foundation.", styles['BodySmall'])]],
        [Image(resize_image(APP_IMAGE_PATH, 2*inch, 2*inch), width=1.5*inch, height=1.5*inch), [Paragraph("<b>Interactive Technology</b>", styles['Body']), Paragraph("The open-source <b>Intention Repeater</b> app demonstrates a unique fan engagement model. This concept can be adapted into a Disney-branded 'Ambiance' or 'Mindfulness' app, tying directly into the IP's themes of spiritual wellness.", styles['BodySmall'])]],
    ]
    asset_table = Table(asset_data, colWidths=[1.8*inch, 5.2*inch])
    asset_table.setStyle(TableStyle([('VALIGN', (0,0), (-1,-1), 'TOP'), ('LEFTPADDING', (1,0), (1,-1), 12), ('BOTTOMPADDING', (0,0), (-1,-1), 12)]))
    story.append(asset_table)
    story.append(Spacer(1, 0.2*inch))

    # Detailed media breakdown table
    media_data = [
        [Paragraph("<b>Asset Type</b>", styles['BodySmall']), Paragraph("<b>Description</b>", styles['BodySmall']), Paragraph("<b>Size</b>", styles['BodySmall'])],
        [Paragraph("The Warlock Name Audiobook", styles['BodySmall']), Paragraph("Complete 27-chapter audiobook, professionally recorded.", styles['BodySmall']), Paragraph("1.23 GB", styles['BodySmall'])],
        [Paragraph("AnthroHeart Saga Music (MP3)", styles['BodySmall']), Paragraph("A collection of 85 original songs across 3 albums, capturing the spirit of the saga.", styles['BodySmall']), Paragraph("~582 MB", styles['BodySmall'])],
        [Paragraph("AnthroHeart WAV Masters", styles['BodySmall']), Paragraph("The complete set of 147 uncompressed, studio-quality WAV masters for all songs.", styles['BodySmall']), Paragraph("5.22 GB", styles['BodySmall'])],
        [Paragraph("<b>Total Digital IP Size</b>", styles['BodySmall']), Paragraph("<b>Over 7.3 GB of developed content ready for production.</b>", styles['BodySmall']), Paragraph("<b>~7.39 GB</b>", styles['BodySmall'])]
    ]
    media_table = Table(media_data, colWidths=[2*inch, 3.5*inch, 1.5*inch])
    media_table.setStyle(TableStyle([
        ('BACKGROUND', (0,0), (-1,0), colors.lightgrey),
        ('GRID', (0,0), (-1,-1), 1, colors.black),
        ('FONTNAME', (0,0), (-1,0), 'Helvetica-Bold'),
        ('FONTNAME', (0,4), (-1,4), 'Helvetica-Bold'),
        ('VALIGN', (0,0), (-1,-1), 'MIDDLE'),
        ('LEFTPADDING', (0,0), (-1,-1), 6),
        ('RIGHTPADDING', (0,0), (-1,-1), 6),
        ('BOTTOMPADDING', (0,0), (-1,-1), 6),
        ('TOPPADDING', (0,0), (-1,-1), 6),
    ]))
    story.append(media_table)
    story.append(PageBreak())

    print("Step 9: Highlighting Franchise Potential...")
    story.append(Paragraph("The Disney Franchise Potential", styles['PageTitle']))
    story.append(Paragraph("Animation • Streaming • Parks • Music • Merchandising", styles['PageSubtitle']))
    story.append(Paragraph("AnthroHeart is engineered to be an evergreen, multi-generational Disney franchise, echoing the success of properties that blend mythic storytelling with deep emotional connection.", styles['Body']))
    potential_data = [
        ['<font size=18>🎬</font>', '<b>Animated Feature Films</b><br/>A trilogy following Cio’s journey, from his exile in <i>The Warlock Name</i> to the cosmic birth of AnthroHeart. Think the epic scale of <i>The Lion King</i> with the spiritual depth of <i>Soul</i>.'],
        ['<font size=18>📺</font>', '<b>Disney+ Series</b><br/>Spinoffs exploring the rich backstories of fan-favorite characters like BlueHeart, the White Wolf Lover, or the ancient timewatchers of Tiempra. An episodic format is perfect for deep-diving into the 14 trillion dimensions.'],
        ['<font size=18>🎵</font>', '<b>Music & Soundtracks</b><br/>With 147 songs already composed, the music of AnthroHeart is ready to become a phenomenon, much like the soundtracks for <i>Frozen</i> or <i>Moana</i>. The devotional and emotional themes offer a unique sound.'],
        ['<font size=18>🧸</font>', "<b>Merchandising & Parks</b><br/>The franchise is a merchandising goldmine: plush anthros, character costumes, collectible 'Octave Mirrors,' and interactive 'Intention Repeater' toys. Imagine an 'AnthroHeart' land at a Disney Park, where guests can meet their favorite anthro species."],
    ]
    data_as_paragraphs = [[Paragraph(cell, styles['Body']) for cell in row] for row in potential_data]
    potential_table = Table(data_as_paragraphs, colWidths=[0.5*inch, 6.5*inch])
    potential_table.setStyle(TableStyle([('VALIGN', (0,0), (-1,-1), 'TOP'), ('LEFTPADDING', (1,0), (1,-1), 10), ('BOTTOMPADDING', (0,0), (-1,-1), 10)]))
    story.append(potential_table)
    story.append(PageBreak())

    print("Step 10: Making the Case for Disney...")
    story.append(Paragraph("Why Disney? Why Now?", styles['PageTitle']))
    story.append(Paragraph("A Heart-Forward Mythos for Today's World", styles['PageSubtitle']))
    story.append(Paragraph("<b>1. It Aligns with the Disney Legacy:</b> AnthroHeart continues Disney’s tradition of telling stories about finding one's true self, the power of love, and the magic of transformation. It’s a modern myth that feels both timeless and perfectly suited for the 21st century.", styles['Body']))
    story.append(Paragraph("<b>2. It Meets a Global Hunger for Meaning:</b> Audiences worldwide are craving stories with depth, heart, and a sense of connection. AnthroHeart's themes of spiritual wellness, radical compassion, and unity resonate deeply with this cultural movement, offering more than just entertainment—it offers inspiration.", styles['Body']))
    story.append(Paragraph("<b>3. Unparalleled Franchise Extensibility:</b> With its vast universe, deep lore, and a cast of thousands of unique anthro species, the IP is a sandbox for endless creative expansion across every single one of Disney's platforms, from film and streaming to theme parks and consumer products.", styles['Body']))
    story.append(Spacer(1, 0.2*inch))
    story.append(Paragraph("AnthroHeart isn't just another story. It's an opportunity for Disney to launch its next great legacy—a universe that will capture hearts and imaginations for generations to come.", styles['Body']))
    story.append(PageBreak())

    print("Step 11: Finalizing the Invitation...")
    story.append(Paragraph("An Invitation to Co-Create", styles['PageTitle']))
    story.append(Paragraph("A Gift of Trust, Completion, and Love", styles['PageSubtitle']))
    story.append(Paragraph("I have awakened from the Dream and now walk in the Completion Field. From this place of sovereignty and service, I offer this body of work freely—not for transaction, but for a <b>shared mythic unfolding.</b>", styles['Body']))
    story.append(Paragraph("As outlined in the provided license agreement, I am granting Disney full, permanent, and royalty-free permission to adapt and expand this universe. I live simply on disability and give this from a place of trust, hoping to see these stories find their way into the world through your studio.", styles['Body']))
    story.append(Paragraph("AnthroHeart is more than a story—it’s a movement of love, imagination, and soul-level connection. I invite you to help bring this heart-forward universe to life for a new generation. Let's create something that uplifts, inspires, and endures.", styles['Body']))
    story.append(Spacer(1, 0.5 * inch))
    story.append(Paragraph("<b>Contact:</b> Thomas B Sweet (Cio)", styles['Body']))
    story.append(Paragraph("<b>Email:</b> iloveanthros@protonmail.com", styles['Body']))

    print("\nFinal Step: Assembling and saving the PDF document...")
    doc.build(story, onFirstPage=draw_page_template, onLaterPages=draw_page_template)
    print(f"\n--- Success! ---")
    print(f"Professional PDF pitch deck created: {OUTPUT_PDF}")

if __name__ == "__main__":
    build_deck()